<?php
/**
 * PostFinance Checkout SDK
 *
 * This library allows to interact with the PostFinance Checkout payment service.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


namespace PostFinanceCheckout\Sdk\Service;

use PostFinanceCheckout\Sdk\ApiClient;
use PostFinanceCheckout\Sdk\ApiException;
use PostFinanceCheckout\Sdk\ApiResponse;
use PostFinanceCheckout\Sdk\Http\HttpRequest;
use PostFinanceCheckout\Sdk\ObjectSerializer;

/**
 * PaymentTerminalTillService service
 *
 * @category Class
 * @package  PostFinanceCheckout\Sdk
 * @author   wallee AG
 * @license  http://www.apache.org/licenses/LICENSE-2.0 Apache License v2
 */
class PaymentTerminalTillService {

	/**
	 * The API client instance.
	 *
	 * @var ApiClient
	 */
	private $apiClient;

	/**
	 * Constructor.
	 *
	 * @param ApiClient $apiClient the api client
	 */
	public function __construct(ApiClient $apiClient) {
		if (is_null($apiClient)) {
			throw new \InvalidArgumentException('The api client is required.');
		}

		$this->apiClient = $apiClient;
	}

	/**
	 * Returns the API client instance.
	 *
	 * @return ApiClient
	 */
	public function getApiClient() {
		return $this->apiClient;
	}


	/**
	 * Operation performTransaction
	 *
	 * Perform Payment Terminal Transaction
	 *
	 * @param int $space_id  (required)
	 * @param int $transaction_id The ID of the transaction which is used to process with the terminal. (required)
	 * @param int $terminal_id The ID of the terminal which should be used to process the transaction. (required)
	 * @param string $language The language in which the messages should be rendered in. (optional)
	 * @throws \PostFinanceCheckout\Sdk\ApiException
	 * @throws \PostFinanceCheckout\Sdk\VersioningException
	 * @throws \PostFinanceCheckout\Sdk\Http\ConnectionException
	 * @return \PostFinanceCheckout\Sdk\Model\Transaction
	 */
	public function performTransaction($space_id, $transaction_id, $terminal_id, $language = null) {
		return $this->performTransactionWithHttpInfo($space_id, $transaction_id, $terminal_id, $language)->getData();
	}

	/**
	 * Operation performTransactionWithHttpInfo
	 *
	 * Perform Payment Terminal Transaction
     * (Time out for this request is 90 seconds.)
     *
	 * @param int $space_id  (required)
	 * @param int $transaction_id The ID of the transaction which is used to process with the terminal. (required)
	 * @param int $terminal_id The ID of the terminal which should be used to process the transaction. (required)
	 * @param string $language The language in which the messages should be rendered in. (optional)
	 * @throws \PostFinanceCheckout\Sdk\ApiException
	 * @throws \PostFinanceCheckout\Sdk\VersioningException
	 * @throws \PostFinanceCheckout\Sdk\Http\ConnectionException
	 * @return ApiResponse
	 */
	public function performTransactionWithHttpInfo($space_id, $transaction_id, $terminal_id, $language = null) {
		// verify the required parameter 'space_id' is set
		if (is_null($space_id)) {
			throw new \InvalidArgumentException('Missing the required parameter $space_id when calling performTransaction');
		}
		// verify the required parameter 'transaction_id' is set
		if (is_null($transaction_id)) {
			throw new \InvalidArgumentException('Missing the required parameter $transaction_id when calling performTransaction');
		}
		// verify the required parameter 'terminal_id' is set
		if (is_null($terminal_id)) {
			throw new \InvalidArgumentException('Missing the required parameter $terminal_id when calling performTransaction');
		}
		// header params
		$headerParams = [];
		$headerAccept = $this->apiClient->selectHeaderAccept(['application/json;charset=utf-8']);
		if (!is_null($headerAccept)) {
			$headerParams[HttpRequest::HEADER_KEY_ACCEPT] = $headerAccept;
		}
		$headerParams[HttpRequest::HEADER_KEY_CONTENT_TYPE] = $this->apiClient->selectHeaderContentType(['*/*']);

		// query params
		$queryParams = [];
		if (!is_null($space_id)) {
			$queryParams['spaceId'] = $this->apiClient->getSerializer()->toQueryValue($space_id);
		}
		if (!is_null($transaction_id)) {
			$queryParams['transactionId'] = $this->apiClient->getSerializer()->toQueryValue($transaction_id);
		}
		if (!is_null($terminal_id)) {
			$queryParams['terminalId'] = $this->apiClient->getSerializer()->toQueryValue($terminal_id);
		}
		if (!is_null($language)) {
			$queryParams['language'] = $this->apiClient->getSerializer()->toQueryValue($language);
		}

		// path params
		$resourcePath = '/payment-terminal-till/perform-transaction';
		// default format to json
		$resourcePath = str_replace('{format}', 'json', $resourcePath);

		// form params
		$formParams = [];
		
		// for model (json/xml)
		$httpBody = '';
		if (isset($tempBody)) {
			$httpBody = $tempBody; // $tempBody is the method argument, if present
		} elseif (!empty($formParams)) {
			$httpBody = $formParams; // for HTTP post (form)
		}
		// make the API Call
		try {
			$response = $this->apiClient->callApi(
				$resourcePath,
				'GET',
				$queryParams,
				$httpBody,
				$headerParams,
				'\PostFinanceCheckout\Sdk\Model\Transaction',
				'/payment-terminal-till/perform-transaction',
                90
            );
			return new ApiResponse($response->getStatusCode(), $response->getHeaders(), $this->apiClient->getSerializer()->deserialize($response->getData(), '\PostFinanceCheckout\Sdk\Model\Transaction', $response->getHeaders()));
		} catch (ApiException $e) {
			switch ($e->getCode()) {
                case 200:
                    $data = ObjectSerializer::deserialize(
                        $e->getResponseBody(),
                        '\PostFinanceCheckout\Sdk\Model\Transaction',
                        $e->getResponseHeaders()
                    );
                    $e->setResponseObject($data);
                break;
                case 409:
                    $data = ObjectSerializer::deserialize(
                        $e->getResponseBody(),
                        '\PostFinanceCheckout\Sdk\Model\ClientError',
                        $e->getResponseHeaders()
                    );
                    $e->setResponseObject($data);
                break;
                case 442:
                    $data = ObjectSerializer::deserialize(
                        $e->getResponseBody(),
                        '\PostFinanceCheckout\Sdk\Model\ClientError',
                        $e->getResponseHeaders()
                    );
                    $e->setResponseObject($data);
                break;
                case 542:
                    $data = ObjectSerializer::deserialize(
                        $e->getResponseBody(),
                        '\PostFinanceCheckout\Sdk\Model\ServerError',
                        $e->getResponseHeaders()
                    );
                    $e->setResponseObject($data);
                break;
                case 543:
                    $data = ObjectSerializer::deserialize(
                        $e->getResponseBody(),
                        '\PostFinanceCheckout\Sdk\Model\ClientError',
                        $e->getResponseHeaders()
                    );
                    $e->setResponseObject($data);
                break;
			}
			throw $e;
		}
	}

	/**
	 * Operation performTransactionByIdentifier
	 *
	 * Perform Payment Terminal Transaction (using TID)
	 *
	 * @param int $space_id  (required)
	 * @param int $transaction_id The ID of the transaction which is used to process with the terminal. (required)
	 * @param string $terminal_identifier The identifier (aka TID) of the terminal which should be used to process the transaction. (required)
	 * @param string $language The language in which the messages should be rendered in. (optional)
	 * @throws \PostFinanceCheckout\Sdk\ApiException
	 * @throws \PostFinanceCheckout\Sdk\VersioningException
	 * @throws \PostFinanceCheckout\Sdk\Http\ConnectionException
	 * @return \PostFinanceCheckout\Sdk\Model\Transaction
	 */
	public function performTransactionByIdentifier($space_id, $transaction_id, $terminal_identifier, $language = null) {
		return $this->performTransactionByIdentifierWithHttpInfo($space_id, $transaction_id, $terminal_identifier, $language)->getData();
	}

	/**
	 * Operation performTransactionByIdentifierWithHttpInfo
	 *
	 * Perform Payment Terminal Transaction (using TID)
     * (Time out for this request is 90 seconds.)
     *
	 * @param int $space_id  (required)
	 * @param int $transaction_id The ID of the transaction which is used to process with the terminal. (required)
	 * @param string $terminal_identifier The identifier (aka TID) of the terminal which should be used to process the transaction. (required)
	 * @param string $language The language in which the messages should be rendered in. (optional)
	 * @throws \PostFinanceCheckout\Sdk\ApiException
	 * @throws \PostFinanceCheckout\Sdk\VersioningException
	 * @throws \PostFinanceCheckout\Sdk\Http\ConnectionException
	 * @return ApiResponse
	 */
	public function performTransactionByIdentifierWithHttpInfo($space_id, $transaction_id, $terminal_identifier, $language = null) {
		// verify the required parameter 'space_id' is set
		if (is_null($space_id)) {
			throw new \InvalidArgumentException('Missing the required parameter $space_id when calling performTransactionByIdentifier');
		}
		// verify the required parameter 'transaction_id' is set
		if (is_null($transaction_id)) {
			throw new \InvalidArgumentException('Missing the required parameter $transaction_id when calling performTransactionByIdentifier');
		}
		// verify the required parameter 'terminal_identifier' is set
		if (is_null($terminal_identifier)) {
			throw new \InvalidArgumentException('Missing the required parameter $terminal_identifier when calling performTransactionByIdentifier');
		}
		// header params
		$headerParams = [];
		$headerAccept = $this->apiClient->selectHeaderAccept(['application/json;charset=utf-8']);
		if (!is_null($headerAccept)) {
			$headerParams[HttpRequest::HEADER_KEY_ACCEPT] = $headerAccept;
		}
		$headerParams[HttpRequest::HEADER_KEY_CONTENT_TYPE] = $this->apiClient->selectHeaderContentType(['*/*']);

		// query params
		$queryParams = [];
		if (!is_null($space_id)) {
			$queryParams['spaceId'] = $this->apiClient->getSerializer()->toQueryValue($space_id);
		}
		if (!is_null($transaction_id)) {
			$queryParams['transactionId'] = $this->apiClient->getSerializer()->toQueryValue($transaction_id);
		}
		if (!is_null($terminal_identifier)) {
			$queryParams['terminalIdentifier'] = $this->apiClient->getSerializer()->toQueryValue($terminal_identifier);
		}
		if (!is_null($language)) {
			$queryParams['language'] = $this->apiClient->getSerializer()->toQueryValue($language);
		}

		// path params
		$resourcePath = '/payment-terminal-till/perform-transaction-by-identifier';
		// default format to json
		$resourcePath = str_replace('{format}', 'json', $resourcePath);

		// form params
		$formParams = [];
		
		// for model (json/xml)
		$httpBody = '';
		if (isset($tempBody)) {
			$httpBody = $tempBody; // $tempBody is the method argument, if present
		} elseif (!empty($formParams)) {
			$httpBody = $formParams; // for HTTP post (form)
		}
		// make the API Call
		try {
			$response = $this->apiClient->callApi(
				$resourcePath,
				'GET',
				$queryParams,
				$httpBody,
				$headerParams,
				'\PostFinanceCheckout\Sdk\Model\Transaction',
				'/payment-terminal-till/perform-transaction-by-identifier',
                90
            );
			return new ApiResponse($response->getStatusCode(), $response->getHeaders(), $this->apiClient->getSerializer()->deserialize($response->getData(), '\PostFinanceCheckout\Sdk\Model\Transaction', $response->getHeaders()));
		} catch (ApiException $e) {
			switch ($e->getCode()) {
                case 200:
                    $data = ObjectSerializer::deserialize(
                        $e->getResponseBody(),
                        '\PostFinanceCheckout\Sdk\Model\Transaction',
                        $e->getResponseHeaders()
                    );
                    $e->setResponseObject($data);
                break;
                case 409:
                    $data = ObjectSerializer::deserialize(
                        $e->getResponseBody(),
                        '\PostFinanceCheckout\Sdk\Model\ClientError',
                        $e->getResponseHeaders()
                    );
                    $e->setResponseObject($data);
                break;
                case 442:
                    $data = ObjectSerializer::deserialize(
                        $e->getResponseBody(),
                        '\PostFinanceCheckout\Sdk\Model\ClientError',
                        $e->getResponseHeaders()
                    );
                    $e->setResponseObject($data);
                break;
                case 542:
                    $data = ObjectSerializer::deserialize(
                        $e->getResponseBody(),
                        '\PostFinanceCheckout\Sdk\Model\ServerError',
                        $e->getResponseHeaders()
                    );
                    $e->setResponseObject($data);
                break;
                case 543:
                    $data = ObjectSerializer::deserialize(
                        $e->getResponseBody(),
                        '\PostFinanceCheckout\Sdk\Model\ClientError',
                        $e->getResponseHeaders()
                    );
                    $e->setResponseObject($data);
                break;
			}
			throw $e;
		}
	}


}
